# 通用项目架构模板

## 1️⃣ Python Web/API 项目标准结构

```
项目名称/
├── README.md                 # 项目说明文档
├── LICENSE                   # 开源协议
├── requirements.txt          # 依赖管理（pip）
├── pyproject.toml           # 现代Python项目配置（推荐）
├── setup.py                 # 包安装脚本（如果做成库）
├── .gitignore              # Git忽略文件
├── .env                    # 环境变量（不提交到Git）
├── .env.example            # 环境变量示例
├── CLAUDE.md              # claude持久上下文
├── AGENTS.md              # codex持久上下文
├── Sublime-Text.txt                   # 放需求和注意事项，给自己看的，和cli的会话恢复指令^_^
│
├── docs/                   # 文档目录
│   ├── api.md             # API文档
│   ├── development.md     # 开发指南
│   └── architecture.md    # 架构说明
│
├── scripts/               # 脚本工具
│   ├── deploy.sh          # 部署脚本
│   ├── backup.sh          # 备份脚本
│   └── init_db.sh         # 数据库初始化
│
├── tests/                 # 测试代码
│   ├── __init__.py
│   ├── conftest.py        # pytest配置
│   ├── unit/              # 单元测试
│   ├── integration/       # 集成测试
│   └── test_config.py     # 配置测试
│
├── src/                   # 源代码（推荐方式）
│   ├── __init__.py
│   ├── main.py           # 程序入口
│   ├── app.py            # Flask/FastAPI应用
│   ├── config.py         # 配置管理
│   │
│   ├── core/             # 核心业务逻辑
│   │   ├── __init__.py
│   │   ├── models/       # 数据模型
│   │   ├── services/     # 业务服务
│   │   └── utils/        # 工具函数
│   │
│   ├── api/              # API接口层
│   │   ├── __init__.py
│   │   ├── v1/           # 版本1
│   │   └── dependencies.py
│   │
│   ├── data/             # 数据处理
│   │   ├── __init__.py
│   │   ├── repository/   # 数据访问层
│   │   └── migrations/   # 数据库迁移
│   │
│   └── external/         # 外部服务
│       ├── __init__.py
│       ├── clients/      # API客户端
│       └── integrations/ # 集成服务
│
├── logs/                  # 日志目录（不提交到Git）
│   ├── app.log
│   └── error.log
│
└── data/                  # 数据目录（不提交到Git）
    ├── raw/               # 原始数据
    ├── processed/         # 处理后的数据
    └── cache/             # 缓存
```

**使用场景**：Flask/FastAPI Web应用、RESTful API服务、Web后端

---

## 2️⃣ 数据科学/量化项目标准结构

```
项目名称/
├── README.md
├── LICENSE
├── requirements.txt
├── .gitignore
├── .env
├── .env.example
├── CLAUDE.md              # claude持久上下文
├── AGENTS.md              # codex持久上下文
├── Sublime-Text.txt                   # 放需求和注意事项，给自己看的，和cli的会话恢复指令^_^
│
├── docs/                   # 文档目录
│   ├── notebooks/         # Jupyter文档
│   └── reports/           # 分析报告
│
├── notebooks/             # Jupyter Notebook
│   ├── 01_data_exploration.ipynb
│   ├── 02_feature_engineering.ipynb
│   └── 03_model_training.ipynb
│
├── scripts/               # 脚本工具
│   ├── train_model.py     # 训练脚本
│   ├── backtest.py        # 回测脚本
│   ├── collect_data.py    # 数据采集
│   └── deploy_model.py    # 模型部署
│
├── tests/                 # 测试
│   ├── test_data/
│   └── test_models/
│
├── configs/               # 配置文件
│   ├── model.yaml
│   ├── database.yaml
│   └── trading.yaml
│
├── src/                   # 源代码
│   ├── __init__.py
│   │
│   ├── data/              # 数据处理模块
│   │   ├── __init__.py
│   │   ├── collectors/    # 数据采集器
│   │   ├── processors/    # 数据清洗
│   │   ├── features/      # 特征工程
│   │   └── loaders.py     # 数据加载
│   │
│   ├── models/            # 模型模块
│   │   ├── __init__.py
│   │   ├── strategies/    # 交易策略
│   │   ├── backtest/      # 回测引擎
│   │   └── risk/          # 风险管理
│   │
│   ├── utils/             # 工具模块
│   │   ├── __init__.py
│   │   ├── logging.py     # 日志配置
│   │   ├── database.py    # 数据库工具
│   │   └── api_client.py  # API客户端
│   │
│   └── core/              # 核心模块
│       ├── __init__.py
│       ├── config.py      # 配置管理
│       ├── signals.py     # 信号生成
│       └── portfolio.py   # 投资组合
│
├── data/                  # 数据目录（Git忽略）
│   ├── raw/               # 原始数据
│   ├── processed/         # 处理后数据
│   ├── external/          # 外部数据
│   └── cache/             # 缓存
│
├── models/                # 模型文件（Git忽略）
│   ├── checkpoints/       # 检查点
│   └── exports/           # 导出模型
│
└── logs/                  # 日志（Git忽略）
    ├── trading.log
    └── errors.log
```

**使用场景**：量化交易、机器学习、数据分析、AI研究

---

## 3️⃣ Monorepo（多项目仓库）标准结构

```
项目名称-monorepo/
├── README.md
├── LICENSE
├── .gitignore
├── .gitmodules           # Git子模块
├── docker-compose.yml    # Docker编排
├── CLAUDE.md              # claude持久上下文
├── AGENTS.md              # codex持久上下文
├── Sublime-Text.txt                   # 这个是文件，放需求和注意事项，给自己看的，和cli的会话恢复指令^_^
│
├── docs/                 # 全局文档
│   ├── architecture.md
│   └── deployment.md
│
├── scripts/              # 全局脚本
│   ├── build_all.sh
│   ├── test_all.sh
│   └── deploy.sh
│
├── backups/                 #  放备份文件
│   ├── archive/             #  放旧的备份文件
│   └── gz/                  #  放备份文件的gz
│
├── services/             # 微服务目录
│   │
│   ├── user-service/     # 用户服务
│   │   ├── Dockerfile
│   │   ├── requirements.txt
│   │   ├── src/
│   │   └── tests/
│   │
│   ├── trading-service/  # 交易服务
│   │   ├── Dockerfile
│   │   ├── requirements.txt
│   │   ├── src/
│   │   └── tests/
│   ...
│   └── data-service/     # 数据服务
│       ├── Dockerfile
│       ├── requirements.txt
│       ├── src/
│       └── tests/
│
├── libs/                 # 共享库
│   ├── common/           # 公共模块
│   │   ├── utils/
│   │   └── models/
│   ├── external/         # 第三方库（不可修改，只调用）
│   └── database/         # 数据库访问库
│
├── infrastructure/       # 基础设施
│   ├── terraform/        # 云资源定义
│   ├── kubernetes/       # K8s配置
│   └── nginx/            # 反向代理配置
│
└── monitoring/           # 监控系统
    ├── prometheus/       # 指标收集
    ├── grafana/          # 可视化
    └── alertmanager/     # 告警
```

**使用场景**：微服务架构、大型项目、团队协作

---

## 4️⃣ Full-Stack Web 应用标准结构

```
项目名称/
├── README.md
├── LICENSE
├── .gitignore
├── docker-compose.yml    # 前后端一起编排
├── CLAUDE.md              # claude持久上下文
├── AGENTS.md              # codex持久上下文
├── Sublime-Text.txt                   # 放需求和注意事项，给自己看的，和cli的会话恢复指令^_^
│
├── frontend/             # 前端目录
│   ├── public/           # 静态资源
│   ├── src/              # 源码
│   │   ├── components/   # React/Vue组件
│   │   ├── pages/        # 页面
│   │   ├── store/        # 状态管理
│   │   └── utils/        # 工具
│   ├── package.json      # NPM依赖
│   └── vite.config.js    # 构建配置
│
└── backend/              # 后端目录
    ├── requirements.txt
    ├── Dockerfile
    ├── src/
    │   ├── api/          # API接口
    │   ├── core/         # 业务逻辑
│   │   └── models/       # 数据模型
    └── tests/
```

**使用场景**：全栈应用、SPA单页应用、前后端分离项目

---

## 📌 核心设计原则

### 1. 关注点分离（Separation of Concerns）
```
API → 服务 → 数据访问 → 数据库
一目了然，层级清晰
```

### 2. 可测试性（Testability）
```
每个模块可独立测试
依赖可mock
```

### 3. 可配置性（Configurability）
```
配置与代码分离
环境变量 > 配置文件 > 默认值
```

### 4. 可维护性（Maintainability）
```
代码自解释
合理的文件命名
清晰的目录结构
```

### 5. 版本控制友好（Git-Friendly）
```
data/、logs/、models/ 添加到 .gitignore
只提交源代码和配置示例
```

---

## 🎯 最佳实践建议

1. **使用 `src/` 目录**：把源代码放在专门的src目录，避免顶级目录混乱
2. **相对导入**：统一使用 `from src.module import thing` 的导入方式
3. **测试覆盖**：保证核心业务逻辑有单元测试和集成测试
4. **文档先行**：重要模块都要写README.md说明
5. **环境隔离**：使用virtualenv或conda创建独立环境
6. **依赖明确**：所有依赖都写入requirements.txt，并锁定版本
7. **配置管理**：使用环境变量 + 配置文件的组合方式
8. **日志分级**：DEBUG、INFO、WARNING、ERROR、FATAL
9. **错误处理**：不要吞掉异常，要有完整的错误链
10. **代码规范**：使用black格式化，flake8检查

---

## 🔥 .gitignore 推荐模板

```gitignore
# Python
__pycache__/
*.py[cod]
*$py.class
*.so
.Python
*.egg-info/
dist/
build/

# 环境
.env
.venv/
env/
venv/
ENV/

# IDE
.vscode/
.idea/
*.swp
*.swo
*~

# 数据
data/
*.csv
*.json
*.db
*.sqlite
*.duckdb

# 日志
logs/
*.log

# 模型
models/
*.h5
*.pkl

# 临时文件
tmp/
temp/
*.tmp
.DS_Store
```

---

## 📚 技术选型参考

| 场景 | 推荐技术栈 |
|-----|----------|
| Web API | FastAPI + Pydantic + SQLAlchemy |
| 数据处理 | Pandas + NumPy + Polars |
| 机器学习 | Scikit-learn + XGBoost + LightGBM |
| 深度学习 | PyTorch + TensorFlow |
| 数据库 | PostgreSQL + Redis |
| 消息队列 | RabbitMQ / Kafka |
| 任务队列 | Celery |
| 监控 | Prometheus + Grafana |
| 部署 | Docker + Docker Compose |
| CI/CD | GitHub Actions / GitLab CI |

---

## 📝 文件模板示例

### requirements.txt
```txt
# 核心依赖
fastapi==0.104.1
uvicorn[standard]==0.24.0
pydantic==2.5.0

# 数据库
sqlalchemy==2.0.23
alembic==1.12.1
psycopg2-binary==2.9.9

# 测试
pytest==7.4.3
pytest-cov==4.1.0
pytest-asyncio==0.21.1

# 工具
python-dotenv==1.0.0
loguru==0.7.2

# 开发（可选）
black==23.11.0
flake8==6.1.0
mypy==1.7.1
```

### pyproject.toml（现代Python项目推荐）
```toml
[project]
name = "项目名称"
version = "0.1.0"
description = "项目描述"
authors = [{name = "作者", email = "邮箱@example.com"}]
dependencies = [
    "fastapi>=0.104.0",
    "uvicorn[standard]>=0.24.0",
    "sqlalchemy>=2.0.0",
]

[project.optional-dependencies]
dev = ["pytest", "black", "flake8", "mypy"]

[build-system]
requires = ["setuptools", "wheel"]
build-backend = "setuptools.build_meta"
```

---

## ✅ 新项目检查清单

启动新项目时，确保完成以下事项：

- [ ] 创建README.md，包含项目简介和使用说明
- [ ] 创建LICENSE文件，明确开源协议
- [ ] 设置Python虚拟环境（venv/conda）
- [ ] 创建requirements.txt并锁定依赖版本
- [ ] 创建.gitignore，排除敏感和不必要的文件
- [ ] 创建.env.example，说明需要的环境变量
- [ ] 设计目录结构，符合关注点分离原则
- [ ] 创建基础的配置文件
- [ ] 设置代码格式化工具（black）
- [ ] 设置代码检查工具（flake8/ruff）
- [ ] 编写第一个测试用例
- [ ] 设置Git仓库并提交初始代码
- [ ] 创建CHANGELOG.md，记录版本变更

---

**版本**: 1.0
**更新日期**: 2025-11-24
**维护**: CLAUDE，CODEX，KIMI
